package net.udooz.azure.jaz;

import org.apache.commons.codec.binary.Base64;
import com.ning.http.client.*;

import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

public class AzureStorage {
	protected final String AccountUrlSuffix = "core.windows.net";
	protected final String HeaderPrefixMS = "x-ms-";
	protected final String HeaderPrefixProperties = "x-ms-prop-";
	protected final String HeaderPrefixMetadata = "x-ms-meta-";
	protected final String HeaderDate = HeaderPrefixMS + "date";
	
	private String storageMedium;	
	private String accountName;
	private byte[] secretKey;
	private String host;
	private Pattern urlAbsolutePathPattern;	
	
	public AzureStorage(String accountName, String storageMedium, String base64SecretKey) {
		this.accountName = accountName;
		this.storageMedium = storageMedium;
		this.host = "core.windows.net";
		secretKey = Base64.decodeBase64(base64SecretKey);		
		
		urlAbsolutePathPattern = Pattern.compile("http(?:s?)://[-a-z0-9.]+/([-a-z0-9]*)/\\?.*");
	}
	
	public Request get(String resourcePath) {
		String RFC1123_PATTERN =
		        "EEE, dd MMM yyyy HH:mm:ss z";
		DateFormat rfc1123Format =
		        new SimpleDateFormat(RFC1123_PATTERN);
		rfc1123Format.setTimeZone(TimeZone.getTimeZone("GMT"));
		
		String url = "https://" + this.accountName + "." + 
					this.storageMedium + "." +  this.host + "/" + resourcePath;
		RequestBuilder builder = new RequestBuilder("GET");
		Request request = builder.setUrl(url)
				.addHeader("content-type", "text/plain")
				.addHeader("content-length", "0")
				.addHeader(HeaderDate, rfc1123Format.format(new Date()))
				.addHeader(HeaderPrefixMS + "version", "2009-09-19")
				.build();
		
		String authHeader = "";
		try {
			authHeader = createAuthorizationHeader(request);
		} catch (InvalidKeyException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (UnsupportedEncodingException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (NoSuchAlgorithmException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		request.getHeaders().add("Authorization", "SharedKey " + this.accountName
			            + ":" + authHeader);
		return request;
	}
	
	private String createAuthorizationHeader(Request request) throws UnsupportedEncodingException, 
		NoSuchAlgorithmException, InvalidKeyException {
		FluentCaseInsensitiveStringsMap headers = request.getHeaders();
		StringBuffer stringToSign = new StringBuffer();
		stringToSign.append(request.getMethod() + "\n");
		stringToSign.append("\n\n0\n\n");
		stringToSign.append(headers.get("content-type").get(0) + "\n");
		stringToSign.append("\n\n\n\n\n\n");
		
		//Canonicalized Headers
        List<String> httpStorageHeaderNameArray = new ArrayList<String>();
		for(String key : headers.keySet()) {		
			if(key.toLowerCase().startsWith(HeaderPrefixMS)) {
				httpStorageHeaderNameArray.add(key.toLowerCase());
			}
		}
		Collections.sort(httpStorageHeaderNameArray);
		for(String key : httpStorageHeaderNameArray)
        {
            stringToSign.append(key + ":" + headers.get(key).get(0) + "\n");
        }
		
		//Canonicalized Resources
		Matcher matcher = urlAbsolutePathPattern.matcher(request.getUrl());
		String absolutePath = "";
		
		if(matcher.find()) {
			absolutePath = matcher.group(1);
		}else {
			throw new IllegalArgumentException("resourcePath");
		}
		
		stringToSign.append("/" + this.accountName + "/" + absolutePath);
		if(absolutePath.length() > 0) stringToSign.append("/");
		
		stringToSign.append("\n");
		
		List<String> paramsArray = new ArrayList<String>();
		for(String key : request.getQueryParams().keySet()) {
			paramsArray.add(key.toLowerCase());			
		}
		Collections.sort(paramsArray);
		for(String key : paramsArray)
        {
			stringToSign.append(key + ":" + request.getQueryParams().get(key).get(0) + "\n");
        }
		
		
		
		byte[] dataToMac = stringToSign.substring(0, stringToSign.length() -1).getBytes("UTF-8");
		
		SecretKeySpec signingKey = new SecretKeySpec(secretKey, "HmacSHA256");
		Mac hmacSha256 = Mac.getInstance("HmacSHA256");
		hmacSha256.init(signingKey);
		byte[] rawMac = hmacSha256.doFinal(dataToMac);
		return Base64.encodeBase64String(rawMac);
	}
}

